<?php

namespace App\Helpers;

use Illuminate\Http\JsonResponse;
use Symfony\Component\HttpFoundation\Response;

/**
 * Utils for working with API error response.
 *
 * @package App\Utils
 */
class ApiUtil
{
    public static function abort(
        string $message,
        array $errors = [],
        int $code = Response::HTTP_BAD_REQUEST
    ): JsonResponse {
        return response()->json([
            'message' => $message,
            'errors' => $errors,
            'failed' => 'true',
            'code' => $code,
        ], $code);
    }

    /**
     * Helper function for returning structured JSON success response.
     *
     * @param string $message Response message
     * @param mixed $data Response data
     * @param int $code HTTP response code
     *
     * @return JsonResponse
     */
    public static function success(
        string $message = 'OK',
        mixed $data = null,
        int $code = Response::HTTP_OK
    ): JsonResponse {
        return response()->json([
            'success' => true,
            'message' => $message,
            'data' => $data,
        ], $code);
    }

    /**
     * Helper function for returning structured JSON failure response.
     *
     * @param string $message Response message
     * @param mixed $data Response data
     * @param int $code HTTP response code
     * @param array $errors response errors
     *
     * @return JsonResponse
     */
    public static function failure(
        string $message = 'ERROR',
        mixed $data = null,
        int $code = Response::HTTP_BAD_REQUEST,
        array $errors = [],
    ): JsonResponse {
        return response()->json([
            'message' => $message,
            'errors' => $errors,
            'data' => $data,
            'failed' => 'true',
            'code' => $code,
        ], $code);
    }

    /**
     * Helper function for returning structured JSON crash response.
     *
     * @param string $message Response message
     * @param mixed $data Response data
     * @param int $code HTTP response code
     * @param array $errors response errors
     *
     * @return JsonResponse
     */
    public static function crash(
        string $message = 'SYSTEM ERROR',
        mixed $data = null,
        int $code = Response::HTTP_INTERNAL_SERVER_ERROR,
        array $errors = [],
    ): JsonResponse {
        return response()->json([
            'message' => $message,
            'errors' => $errors,
            'failed' => 'true',
            'code' => $code,
        ], $code);
    }

    /**
     * Helper function for returning structured JSON not found response.
     *
     * @param string $message Response message
     * @param mixed $data Response data
     * @param int $code HTTP response code
     * @param array $errors response errors
     *
     * @return JsonResponse
     */
    public static function notFound(
        string $message = 'NOT FOUND',
        mixed $data = null,
        int $code = Response::HTTP_NOT_FOUND,
        array $errors = [],
    ): JsonResponse {
        return response()->json([
            'message' => $message,
            'errors' => $errors,
            'failed' => 'true',
            'code' => $code,
        ], $code);
    }

    /**
     * Helper function for returning structured JSON service unavailable response.
     *
     * @param string $message Response message
     * @param mixed $data Response data
     * @param int $code HTTP response code
     *
     * @return JsonResponse
     */
    public static function unavailable(
        string $message = 'SERVICE UNAVAILABLE',
        mixed $data = null,
        int $code = Response::HTTP_NOT_IMPLEMENTED,
    ): JsonResponse {
        return response()->json([
            'success' => false,
            'message' => $message,
            'data' => $data,
        ], $code);
    }
}
